<?php

namespace Meibuyu\Micro\Middleware;

use FastRoute\Dispatcher;
use Hyperf\HttpServer\Router\DispatcherFactory;
use Hyperf\Utils\ApplicationContext;
use Meibuyu\Micro\Model\Auth;
use Meibuyu\Micro\Service\Interfaces\User\AuthenticationServiceInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\MiddlewareInterface;
use Psr\Http\Server\RequestHandlerInterface;
use Hyperf\Di\Annotation\Inject;

class AuthorizeMiddleware implements MiddlewareInterface
{

    /**
     * @Inject()
     * @var AuthenticationServiceInterface
     */
    private $authorizationService;

    public function process(ServerRequestInterface $request, RequestHandlerInterface $handler): ResponseInterface
    {
        $path = $request->getUri()->getPath();
        $token = token();
        $applicationName = env('APP_NAME');
        $method = $request->getMethod();

        if (empty($path)) return $handler->handle($request);
        //获取对应的 path 对应的权限，如果 path 是不需要登录鉴权，直接返回
        $passed  = $this->authRouter($applicationName, $path, $method, $token);
        if ($passed)  {
            return $handler->handle($request);
        }

        return response()->withStatus(403)->json(
            [
                'code' => 403,
                'msg' => "您没有访问接口的权限，请检查后再操作"
            ]);     //鉴权失败，错误码 403 forbidden
        //path 是需要登录鉴权的，判断当前用户是佛有对应  path 的权限

    }


    /**
     * 获取对应路由的权限，调用 RPC 服务
     * @param $applicationName
     * @param $route
     * @param $token
     * @return bool
     */
    protected function authRouter($applicationName, $path, $method, $token):  bool
    {
        $userId = $this->getUserIdByToken($token);
        $route = $this->getRouterByPath($path, $method);
        if (empty($route)) return true;     //说明没有匹配到路由，直接 pass，后续执行一定会返回 404， 这里也可以直接  返回 404

        return $this->authorizationService->authByRouter($applicationName, $route, $method, $userId);
    }

    /**
     * 根据 path 和 method 获取对应的 router
     * @param string $path
     * @param string $method
     * @return array|string
     */
    private function getRouterByPath(string $path, string $method) : string
    {
        $factory = ApplicationContext::getContainer()->get(DispatcherFactory::class);
        $dispatcher = $factory->getDispatcher('http');
        $routerMatched = $dispatcher->dispatch($method, $path);
        $founded =  $routerMatched[0];
        if ( $founded != Dispatcher::FOUND) return '';        //说明没有匹配上路由，可以直接 return 404 not found

        $handler = $routerMatched[1];
        return $handler->route;
    }

    /**
     * 根据 token 获取对应的 user_id
     * @param $token
     * @return int|mixed
     */
    protected function getUserIdByToken($token)
    {
        if (empty($token)) return 0;
        $user = redis()->get($token);
        if ( ! $user)  return 0;

        $userArr =   \json_decode($user, true);
        return !empty($userArr['id']) ? $userArr['id'] : 0;
    }
}