<?php

declare(strict_types=1);

namespace Meibuyu\Micro\Exceptions\Handler;

use Exception;
use Hyperf\Amqp\Producer;
use Hyperf\Contract\ConfigInterface;
use Hyperf\Contract\StdoutLoggerInterface;
use Hyperf\ExceptionHandler\ExceptionHandler;
use Hyperf\HttpMessage\Stream\SwooleStream;
use Hyperf\Logger\LoggerFactory;
use Meibuyu\Micro\Amqp\Producer\ExceptionLogProducer;
use Psr\Container\ContainerInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Throwable;

class AppExceptionHandler extends ExceptionHandler
{

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var StdoutLoggerInterface
     */
    protected $stdoutLogger;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var ConfigInterface
     */
    protected $config;

    public function __construct(ContainerInterface $container)
    {
        $this->container = $container;
        $this->stdoutLogger = $container->get(StdoutLoggerInterface::class);
        $this->logger = $container->get(LoggerFactory::class)->get('Uncaught Exception');
        $this->config = $container->get(ConfigInterface::class);
    }

    public function handle(Throwable $throwable, ResponseInterface $response)
    {
        // 捕获所有未捕获的异常
        $this->stopPropagation();
        $message = $throwable->getMessage();
        $line = $throwable->getLine();
        $file = $throwable->getFile();
        $trace = $throwable->getTraceAsString();
        $code = $throwable->getCode();
        $data = [
            'server' => $this->config->get('app_name'),
            'file' => $file,
            'line' => $line,
            'message' => $message,
            'trace' => $trace,
            'code' => $code,
            'created_at' => now(),
        ];
        try {
            $exceptionLogProducer = new ExceptionLogProducer($data);
            $producer = $this->container->get(Producer::class);
            $producer->produce($exceptionLogProducer);
        } catch (Exception $e) {
            put_log('异常日志失败; ' . $e->getMessage(), 'ExceptionLogProducer.log');
        }
        $msg = sprintf('%s[%s] in %s', $message, $line, $file);
        $this->stdoutLogger->error($msg);
        $this->logger->error($msg);
        $this->stdoutLogger->error($trace);
        return $response->withStatus(500)->withBody(new SwooleStream($msg));
    }

    public function isValid(Throwable $throwable): bool
    {
        return true;
    }
}
