<?php

namespace Meibuyu\Micro\CenterConfig\Nacos;


use GuzzleHttp\Client;
use GuzzleHttp\RequestOptions;
use Meibuyu\Micro\CenterConfig\Driver\DriverInterface;
use Psr\Http\Message\ResponseInterface;

class NacosDriver implements DriverInterface
{
    
    private Config $config;

    private $accessToken;

    private $expireTime;

    /**
     * Notes: 拉取配置文件
     * User: carlos
     * DateTime: 2022/9/9 10:13
     * @param array $config
     */
    public function fetchConfig(array $config)
    {
        $envFileName = BASE_PATH.'/.env';
        //配置nacos配置类
        $this->configure($config);
        if($config['APP_ENV'] == 'dev') {
            $dataId = $config['APP_NAME'];
            $group = $config['APP_ENV'];
            $tenant = $config['APP_ENV'];
        }else{
            $dataId = $config['ENV_DATA_ID'];
            $group = $config['ENV_GROUP'];
            $tenant = $config['ENV_TENANT'];
        }

        $res = $this->get($dataId, $group, $tenant);

        $config['CONFIG_CENTER'] = 'false';
        file_put_contents($envFileName, '');
        foreach ($config as $item=>$value) {
            $content = $item.'='.$value.PHP_EOL;
            file_put_contents($envFileName, $content, FILE_APPEND);
        }
        file_put_contents($envFileName, ''.PHP_EOL, FILE_APPEND);
        file_put_contents($envFileName, $res, FILE_APPEND);
    }

    /**
     * Notes: 获取配置文件
     * User: carlos
     * DateTime: 2022/9/9 13:58
     * @param string $dataId
     * @param string $group
     * @param string|null $tenant
     * @throws \Exception
     */
    public function get(string $dataId, string $group, ?string $tenant = null)
    {
        $res =  $this->request('GET', '/nacos/v1/cs/configs', [
            RequestOptions::QUERY => $this->filter([
                'dataId' => $dataId,
                'group' => $group,
                'tenant' => $tenant,
            ]),
        ]);


        $result = $this->handleResponse(
            $res, 'string'
        );
        return $result;
    }


    protected function filter(array $input): array
    {
        $result = [];
        foreach ($input as $key => $value) {
            if ($value !== null) {
                $result[$key] = $value;
            }
        }

        return $result;
    }


    public function request($method, $uri, array $options = [])
    {
        $token = $this->getAccessToken();
        $token && $options[RequestOptions::QUERY]['accessToken'] = $token;
        return $this->getClient()->request($method, $uri, $options);
    }

    /**
     * Notes: 获取token
     * User: carlos
     * DateTime: 2022/9/9 10:40
     */
    public function getAccessToken()
    {
        $username = $this->config->getUsername();
        $password = $this->config->getPassword();

        if (! $this->isExpired()) {
            return $this->accessToken;
        }

        $result = $this->handleResponse(
            $this->login($username, $password)
        );
        $this->accessToken = $result['accessToken'];
        $this->expireTime = $result['tokenTtl'] + time();

        //return $this->accessToken;
    }

    /**
     * Notes: 查看token是否过期
     * User: carlos
     * DateTime: 2022/9/9 11:01
     * @return bool
     */
    protected function isExpired(): bool
    {
        if (isset($this->accessToken) && $this->expireTime > time() + 60) {
            return false;
        }
        return true;
    }

    /**
     * Notes: nacos登录
     * User: carlos
     * DateTime: 2022/9/9 10:47
     */
    public function login($username, $password) : ResponseInterface
    {
        return $this->getClient()->request('POST', '/nacos/v1/auth/users/login', [
            RequestOptions::QUERY => [
                'username' => $username,
            ],
            RequestOptions::FORM_PARAMS => [
                'password' => $password,
            ],
        ]);
    }

    /**
     * Notes: 解析请求
     * User: carlos
     * DateTime: 2022/9/9 10:47
     * @param ResponseInterface $response
     * @return array
     */
    protected function handleResponse(ResponseInterface $response, $type = 'array')
    {
        $statusCode = $response->getStatusCode();
        $contents = (string) $response->getBody();
//        if ($statusCode !== 200) {
//            throw new \Exception($contents, $statusCode);
//        }
        if($type == 'string') {
            return $contents;
        }else{
            return json_decode($contents, true);
        }
    }



    /**
     * Notes: 配置nacos配置
     * User: carlos
     * DateTime: 2022/9/9 10:32
     * @param array $config
     */
    public function configure(array $config)
    {
        $this->config = new Config($config);
    }

    /**
     * Notes: 获取连接
     * User: carlos
     * DateTime: 2022/9/9 10:14
     */
    public function getClient()
    {
        $config = array_merge($this->config->getGuzzleConfig(), [
            'base_uri' => $this->config->getBaseUri(),
        ]);
        return new Client($config);
    }
}