<?php
/**
 * Created by PhpStorm.
 * User: Zero
 * Date: 2020/7/27
 * Time: 11:44
 */

namespace Meibuyu\Micro\Tools;

use Exception;
use Hyperf\Contract\ConfigInterface;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class Drawer
{

    /**
     * @var ConfigInterface
     */
    protected $config;

    private $rootPath;

    private $savePath = '';

    public function __construct()
    {
        $this->config = container(ConfigInterface::class);
        $this->rootPath = $this->config->get('server.settings.document_root', BASE_PATH . '/public');
    }

    private function parseImagePath($imagePath)
    {
        if (strstr($imagePath, 'http') !== false) {
            $imagePath = $this->downloadWebImage($imagePath);
        }
        return $imagePath;
    }

    /**
     * @param string $url
     * @param string $path
     * @return string
     */
    public function downloadWebImage($url, $path = null)
    {
        $url = $this->parseUrl($url);
        // excel画图中下载图片时对图片名做urlencode处理,防止中文名不能正常画图片的bug
        $filename = trim(pathinfo($url, PATHINFO_FILENAME));
        $ext = $this->getImgExt($url);
        $filename = "$filename.$ext";

        $path = $this->rootPath . '/download/images/' . ($path ?: $this->savePath);
        if (!is_dir($path)) {
            // 判断路径是否存在,不存在,则创建
            mkdir($path, 0777, true);
        }
        $filePath = $path . '/' . $filename;
        if (!file_exists($filePath)) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // 信任任何证书
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);
            $file = curl_exec($ch);
            curl_close($ch);
            $resource = fopen($filePath, 'a');
            fwrite($resource, $file);
            fclose($resource);
        }
        return $filePath;
    }

    private function parseUrl($url)
    {
        if (!preg_match('/%[0-9A-Z]{2}/', $url)) {
            $url = rawurlencode($url);
            $url = str_replace("%3A", ":", $url);
            $url = str_replace("%2F", "/", $url);
            $url = str_replace("%3F", "?", $url);
            $url = str_replace("%3D", "=", $url);
        }
        return $url;
    }

    /**
     * @param $imgPath
     * @param int $px
     * @return bool|string
     * @throws Exception
     */
    public function addBoard($imgPath, $px = 2)
    {
        $imgPath = $this->parseImagePath($imgPath);

        $imgPathInfo = pathinfo($imgPath);
        $filename = $imgPathInfo['filename']; // 图片名称
        $ext = $this->getImgExt($imgPath);

        [$img_w, $img_h] = getimagesize($imgPath); // 图片大小

        $savePath = $this->rootPath . '/download/board/' . $this->savePath;
        if (!is_dir($savePath)) {
            // 判断路径是否存在,不存在,则创建
            mkdir($savePath, 0777, true);
        }
        $imgNewPath = $savePath . '/' . $filename . '.' . $ext;
        if (file_exists($imgNewPath)) {
            return $imgNewPath;
        }

        switch ($ext) {
            case 'jpeg':
            case 'jpg':
                $img_create_func = 'imagecreatefromjpeg';
                $img_save_func = 'imagejpeg';
                break;
            case 'png':
                $img_create_func = 'imagecreatefrompng';
                $img_save_func = 'imagepng';
                break;
            case 'bmp':
                $img_create_func = 'imagecreatefrombmp';
                $img_save_func = 'imagebmp';
                break;
            case 'gif':
                $img_create_func = 'imagecreatefromgif';
                $img_save_func = 'imagegif';
                break;
            case 'vnd.wap.wbmp':
                $img_create_func = 'imagecreatefromwbmp';
                $img_save_func = 'imagewbmp';
                break;
            case 'xbm':
                $img_create_func = 'imagecreatefromxbm';
                $img_save_func = 'imagexbm';
                break;
            default:
                throw new Exception("图片类型不支持");
        }

        // 黑色背景图片
        $im = @imagecreatetruecolor(($img_w + $px), ($img_h + $px)) or die ("Cannot Initialize new GD image stream");
        // 为真彩色画布创建背景，再设置为透明
        $color = imagecolorallocate($im, 0, 0, 0);
        imagefill($im, 0, 0, $color);
        imageColorTransparent($im, $color);
        // 把图片放到黑色背景图片上。边框是1px
        $resource = $img_create_func($imgPath);
        imagecopy($im, $resource, $px / 2, $px / 2, 0, 0, $img_w, $img_h);

        $res = $img_save_func($im, $imgNewPath);

        imagedestroy($im);
        return $res ? $imgNewPath : false;
    }

    /**
     * 拼图
     * @param $imgPathList
     * @return bool|string
     */
    public function mergeImages($imgPathList)
    {
        $maxW = $maxH = 0;
        $filenameList = [];
        $imageList = [];
        foreach ($imgPathList as $k => $path) {
            $path = $this->parseImagePath($path);
            $imgPathInfo = pathinfo($path);
            $filename = $imgPathInfo['filename']; // 图片名称
            $ext = $imgPathInfo['extension']; // 图片扩展名
            [$w, $h] = getimagesize($path); // 图片大小
            $imageList[$k] = [
                'path' => $path,
                'filename' => $filename,
                'ext' => $ext,
                'w' => $w,
                'h' => $h,
            ];
            $filenameList[] = $filename;
            $maxW += $w;
            if ($maxH < $h) {
                $maxH = $h;
            }
        }
        $filenameList = collect($filenameList)->unique()->sort()->toArray();
        $filename = implode('_', $filenameList);
        $savePath = $this->rootPath . '/download/merge/' . $this->savePath;
        if (!is_dir($savePath)) {
            // 判断路径是否存在,不存在,则创建
            mkdir($savePath, 0777, true);
        }
        $imgNewPath = $savePath . '/' . $filename . '.png';
        if (file_exists($imgNewPath)) {
            return $imgNewPath;
        }
        // 黑色背景图片
        $im = @imagecreatetruecolor($maxW, $maxH) or die ("Cannot Initialize new GD image stream");
        // 为真彩色画布创建背景，再设置为透明
        $color = imagecolorallocate($im, 0, 0, 0);
        imagefill($im, 0, 0, $color);
        imageColorTransparent($im, $color);
        // 循环画图片
        $dstX = 0;
        foreach ($imageList as $item) {
            $resource = imagecreatefromstring(file_get_contents($item['path']));
            imagecopy($im, $resource, $dstX, 0, 0, 0, $item['w'], $item['h']);
            $dstX += $item['w'];
        }
        $res = imagepng($im, $imgNewPath);
        imagedestroy($im);
        return $res ? $imgNewPath : false;
    }

    /**
     * 画图片
     * @param string $path 图片路径
     * @param int $h 图片高度
     * @param string $p 单元格索引
     * @param Worksheet $sheet
     * @param int $boardPx
     */
    public function draw2Excel($path, $h, $p, $sheet, $boardPx = 0)
    {
        try {
            $path = $this->parseImagePath($path);
            if ($boardPx) {
                $path = $this->addBoard($path, $boardPx);
            }
            $drawing = new Drawing();
            $drawing->setPath($path)->setCoordinates($p)->setHeight($h)->setOffsetX(1)->setOffsetY(1)->setWorksheet($sheet);
        } catch (Exception $e) {
            put_log($e->getMessage());
        }
    }

    public function setSavePath($path)
    {
        $this->savePath = $path;
    }

    /**
     * 获取图片扩展名
     * @param $path
     * @return bool|mixed|string
     * @author Zero
     */
    public function getImgExt($path)
    {
        $ext = false;
        $info = getimagesize($path);
        if (isset($info['mime']) && $info['mime']) {
            $ext = explode('/', $info['mime'])[1];
        } else {
            $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
            $ext = explode('?', $ext)[0] ?? $ext;
        }
        return $ext;
    }

}
